/**
 * @fileOverview  ZoomRotation - new build microscope
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
OU.require('OU.util.Slider');
OU.require('OU.util.Layer');
OU.require('OU.util.TileRotation');
OU.require('OU.util.ImageLoader');

/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.ZoomRotation = function ( data, instance, controller ) {
    /**
     * Entry point for canvas supporting platforms
     */
    OU.activity.ZoomRotation.prototype.canvasView = function () {
        var self = this, buttonW, bH = OU.controlHeight;
        // create Canvas Layers & Contexts
        this.bgLayer = new OU.util.Layer({
            container:this,
            id:'bg'
        });
        if (this.data.backgroundColour!==undefined)
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            }); // use specified background colour
        else
            this.bgLayer.context.gradRect(); // use default background
        this.scopeLayer = new OU.util.Layer({
            container:this,
            id:'canvas',
            h:this.h - bH,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
        this.controlsBackLayer = new OU.util.Layer({
            container:this,
            id:'oucontrols',
            y:this.h - bH,
            h:bH,
            hasEvents:true
        });
        this.controlsBackLayer.context.gradRect({
            col1:"#ddd",
            col2:"#ddd"
        });
        this.controlsLayer = new OU.util.Layer({
            container:this,
            id:'oucontrols',
            y:this.h - bH,
            h:bH,
            hasEvents:true
        });
        buttonW = this.w / 10;
        this.rightLayer = new OU.util.Layer({
            container:this,
            id:'rightButton',
            x:this.x + this.w - buttonW - 10,
            y:this.y + this.h - bH - buttonW - 10,
            w:buttonW,
            h:buttonW,
            hasEvents:true
        });
        this.leftLayer = new OU.util.Layer({
            container:this,
            id:'leftButton',
            x:this.x + 10,
            y:this.y + this.h - bH - buttonW - 10,
            w:buttonW,
            h:buttonW,
            hasEvents:true
        });
        // If graphics haven't been defined, then set to defaults
        if(this.data.graphics===undefined) {
            this.data.graphics = {
                "rotateLeft" : {
                    fileName: "object-pointer-left.png"
                },
                "rotateRight" : {
                    fileName: "object-pointer-right.png"
                }
            };
        }
        // Load graphics then start the activity
        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.data.graphics,
            onLoad: function () {
                self.start();
            }
        });        
        
    };        
    OU.activity.ZoomRotation.prototype.start = function () {
        var self=this, bH=OU.controlHeight;
        this.inRotationView = false;
        this.tileView = new OU.util.TileRotation({

            image: {
                w:this.data.width,
                h:this.data.height
            },
            tileSize: this.data.tileSize,
            useTileGroups: true, // this.data.useTileGroups,
            imageBase:this.data.images,
            zoomLevels:this.data.zoomLevels,
            renderAngle:this.data.renderAngle,
            window:{
                w:this.w,
                h:this.h - bH
            },
            container:this,
            context:this.scopeLayer.context,
            zoomCallback:function ( z ) {
                self.setSlider(z);
            }
        });
        this.scopeLayer.events.clickable.push(this.tileView);
        this.initControls();
        this.rotationDirection = 0;
        this.rotate();
    };
    OU.activity.ZoomRotation.prototype.rotate = function () {
        var self = this;
        if (this.rotationDirection > 0)
            this.tileView.changeImage(this.tileView.imageIdx + 1);
        if (this.rotationDirection < 0)
            this.tileView.changeImage(this.tileView.imageIdx - 1);
        setTimeout(function () {
            self.rotate();
        }, 40);
    };
    OU.activity.ZoomRotation.prototype.resize = function () {
        OU.activity.ZoomRotation.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, buttonW = this.w / 10, ctx = this.scopeLayer.context;
        this.bgLayer.resize();
        if (this.data.backgroundColour!==undefined)
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            }); // use specified background colour
        else
            this.bgLayer.context.gradRect(); // use default background
        this.scopeLayer.resize({
            h:this.h - bH
        });
        ctx.font = '18px ' + OU.theme.font;
        ctx.lineWidth = 2;
        ctx.strokeStyle = '#c00';
        ctx.fillStyle = '#c00';
        ctx.textAlign = 'center';
        this.controlsBackLayer.resize({
            y:this.h - bH,
            h:bH
        });
        this.controlsBackLayer.context.gradRect({
            col1:"#ddd",
            col2:"#ddd"
        });
        this.controlsLayer.resize({
            y:this.h - bH,
            h:bH
        });
        this.tileView.window = {
            w:this.w,
            h:this.h - bH
        };
        this.zoomSlider.resize({
            x:bH,
            y:0,
            w:this.w - bH * 2,
            h:bH
        });
        this.rightLayer.resize({
            x:this.x + this.w - buttonW - 10,
            y:this.y + this.h - bH - buttonW - 10,
            w:buttonW,
            h:buttonW
        });
        this.leftLayer.resize({
            x:this.x + 10,
            y:this.y + this.h - bH - buttonW - 10,
            w:buttonW,
            h:buttonW
        });
        this.renderControls();
        this.tileView.doRender=true;
    };
    OU.activity.ZoomRotation.prototype.initControls = function () {
        var ctx = this.controlsLayer.context, bH = OU.controlHeight, self = this,
        URLVars = OU.getUrlVars(),
        initZoom = URLVars['zoom']===undefined ? 0 : parseFloat(URLVars['zoom']),
        clickable = this.controlsLayer.events.clickable;
        clickable.length = 0;
        this.zoomSlider = new OU.util.Slider({
            container:this,
            sliderPos:initZoom,
            instance:'zoom',
            x:bH,
            y:0,
            w:this.w - bH * 2,
            h:bH,
            sliderHeight:bH / 2,
            drawContainer:false,
            callback:function ( val ) {
                self.setZoom(val);
            },
            background:{
                clear:true
            },
            context:ctx
        });
        clickable.push(this.zoomSlider);
        this.renderSlider();
        this.rightLayer.events.clickable.push({
            isHit:function ( x, y, state ) {
                if (state) {
                    self.rotationDirection = -1;
                    self.tileView.disableLoading();
                }
                else {
                    self.rotationDirection = 0;
                    self.tileView.enableLoading();
                }
            }
        });
        this.leftLayer.events.clickable.push({
            isHit:function ( x, y, state ) {
                if (state) {
                    self.rotationDirection = 1;
                    self.tileView.disableLoading();
                }
                else {
                    self.rotationDirection = 0;
                    self.tileView.enableLoading();
                }
            }
        });
        this.renderControls();
    };
    OU.activity.ZoomRotation.prototype.renderControls = function () {
        var buttonW = this.w / 10,
        ctx = this.rightLayer.context,
        img = this.data.graphics.rotateRight.image;

        ctx.drawImage(img,0,0,img.width,img.height);

        ctx = this.leftLayer.context;
        img = this.data.graphics.rotateLeft.image;
        ctx.drawImage(img,buttonW-img.width,0,img.width,img.height);
    };
    OU.activity.ZoomRotation.prototype.renderSlider = function () {
        var bH = OU.controlHeight, self = this;
        this.controlsLayer.context.clearRect(bH * 2 * this.data.images.length, 0, this.w - bH * 2 * (1 + this.data.images.length), bH);
        this.zoomSlider.render();
        setTimeout(function () {
            self.renderSlider();
        }, 40);
    };
    OU.activity.ZoomRotation.prototype.setZoom = function ( val ) { // Called when scrubBar is moved
        this.zoomSlider.render();
        this.tileView.scale(val);
    };
    OU.activity.ZoomRotation.prototype.setSlider = function ( val ) { // Called when TileViewer changed via mousewheel, etc.
        this.zoomSlider.setTarget(val);
    };
    OU.activity.ZoomRotation.prototype.pinch = function ( e, s, x, y, dx, dy, me ) { // called when pinch event detected
        var ns = ((e - s) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x:x,
            y:y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.ZoomRotation, OU.util.Activity);

